/**
 *
 * \file        cresnet_host.cpp
 *
 * \brief       low level uart hardware class as a foundation for CresnetTask class
 *
 * \author      Gennady
 *
 * \date        8/24/2008
 *
 */
#define _CRESNET_HOST_CPP_

#include "cresnet_slave.h"
#include "cresnet_host.hpp"
#include "cbuffer.h" // for FALSE
#include "dm_netmapper.hpp"
#include "memorymanager.h"
#include "product.h"                    // for FirmwareGetName
#include "StreamMgr.h"
#include "field_debug.h"
#include "DmEncapChecksumHelper.h"
#include "hardware.h"

#define MASTER_POLL_TIME 40 //time in msec between start of polls
#define POLL_AGAIN_IMMEDIATE 0xff
#define MASTER_RX_ENCAP_OFFSET 3

#define WAIT_FOR_STOP_POLL_RATE 	                            10
#define WAIT_FOR_STOP_POLL_TIME 	                            (2000/WAIT_FOR_STOP_POLL_RATE)//time in msec to wait for class to process the stop polling call
#define CRESNET_HOST_RECONFIGURE_SEMAPHORE_TIMEOUT_MS           1000

void ManageDMNET_LinkLed(void);

UINT8 CDMCresnetUartHost::m_iTotalDmHosts = 0;

// global class pointers cresnet master classes
CDMCresnetUartHost *g_pDmMaster[TOTAL_NUMBER_OF_DM_MASTERS] = {0, 0};

FN_MASTERPROCESSPACKET pfMasterProcessPacket = 0;
FN_MASTERONLINESTATUS pfMasterOnlineStatus = 0;

void CDMCresnetUartHost::DmHostTimerISR(void)
{
    m_pTimer->ResetTimer();

    // short-circuit when disabled
    if (!m_bEnabled)
        return;

    switch ( m_iNetMode )
    {
        // new delay before packet state
    case DM_MASTER_STATE_BEFORE_SENDING_DATA :
        m_iNetMode = DM_MASTER_STATE_SENDING_DATA ;
        m_pUart->GrabBus() ;
        m_pUart->WriteTxBuffer(m_pTxPacket.data[m_iTxCurrent++]);
        m_pTimer->SetTimer(ONE_CRESNET_CHAR_TIME);
        break;

    case DM_MASTER_STATE_SENDING_DATA:
        m_pUart->WriteTxBuffer(m_pTxPacket.data[m_iTxCurrent++]);
        if (m_iTxCurrent == m_iTxCount) {
            m_pTimer->SetTimer(ONE_CRESNET_CHAR_TIME);
            m_iNetMode = DM_MASTER_STATE_INTERPACKET_DELAY ;
        }
        else {                      // more to send
            m_pTimer->SetTimer(ONE_CRESNET_CHAR_TIME);
        }
        break;

    case DM_MASTER_STATE_INTERPACKET_DELAY:
        m_iNetMode = DM_MASTER_STATE_IDLE ;
//        m_pEvent->Send(0, DMH_EVENT_NONE);
        ReleaseCrestSema() ;
        break;

        /*
    case DM_MASTER_STATE_IDLE :
        ResetState() ;
        m_pEvent->Send(0, DMH_EVENT_NONE);
        ReleaseCrestSema() ;
        break;
    */

    case DM_MASTER_STATE_DELAY_BEFORE_TOKEN_ID :
        m_iNetMode = DM_MASTER_STATE_SENDING_TOKEN_ID;
        m_pUart->WriteTxBuffer(m_iCurrent);
        m_pTimer->SetTimer(ONE_CRESNET_CHAR_TIME);
        break;

    case DM_MASTER_STATE_SENDING_TOKEN_ID :
            m_iNetMode = DM_MASTER_STATE_SENDING_TOKEN_LEN;
            m_pUart->WriteTxBuffer(0);
            m_pTimer->SetTimer(ONE_AND_HALF_CHAR_TIME);
            break;

    case DM_MASTER_STATE_SENDING_TOKEN_LEN :  // 1.5 char length passed, release bus and wait for response
            m_iNetMode = DM_MASTER_STATE_WAIT_FOR_RESPONSE ;
            if (m_iPollState != POLL_STAT_TEST) //Normal mode
            {
                m_pUart->DropBus() ;
            }
            m_pTimer->SetTimer(TWELVE_CRESNET_CHAR_TIME);  // wait_for_response_delay  //
            break;

        // non PPN(regular) states
    case DM_MASTER_STATE_INTERPACKET_DELAY_AFTER_NONEMPTY_TOKEN :
            m_iNetMode = DM_MASTER_STATE_IDLE ;
            m_pEvent->Send(0, DMH_EVENT_DATA_RECEIVED);
            break;

    case DM_MASTER_STATE_INTERPACKET_DELAY_AFTER_EMPTY_TOKEN :
            m_iNetMode = DM_MASTER_STATE_IDLE ;
            m_pEvent->Send(0, DMH_EVENT_EMPTY_TOKEN);
            break;
/*
        case DM_MASTER_STATE_WAIT_INTERPACKET_DELAY_END :
            m_iNetMode = DM_MASTER_STATE_IDLE ;
            m_pEvent->Send(0);
            break;
*/
    case DM_MASTER_STATE_WAIT_FOR_RX :        // error - no bytes received in 10 char time
    case DM_MASTER_STATE_WAIT_FOR_RESPONSE :  // error condition, device is not responding, issue sync
    case DM_MASTER_STATE_RECIEVING_LEN :
    case DM_MASTER_STATE_RECIEVING_DATA :
    case DM_MASTER_STATE_WAIT_FOR_TOKEN_ID :
    case DM_MASTER_STATE_WAIT_FOR_TOKEN_LEN :
        m_pUart->GrabBus() ;
        m_iNetMode = DM_MASTER_STATE_WAIT_BEFORE_SENDING_SYNC_LOW_AFTER_NO_RESPONSE ;
        m_iRxCount = 0 ;
        m_iRxExpected = -1 ;
        m_pTimer->SetTimer(ONE_CRESNET_CHAR_TIME);
        break;

    case DM_MASTER_STATE_WAIT_BEFORE_SENDING_SYNC_LOW_AFTER_NO_RESPONSE:
        m_iNetMode = DM_MASTER_STATE_SENDING_SYNC_LOW_AFTER_NO_RESPONSE ;
        m_iRxCount = 0 ;
        m_iRxExpected = -1 ;
        m_pUart->WriteBaudRate(19200);// set baud to 19200
        m_pUart->WriteTxBuffer(0xf0);
        m_pTimer->SetTimer(TWO_CRESNET_CHAR_TIME);
        break;

    case DM_MASTER_STATE_SENDING_SYNC_LOW_AFTER_NO_RESPONSE :
        m_pUart->GrabBus();
        m_pUart->WriteBaudRate(38400);// restore baud rate
        m_iNetMode = DM_MASTER_STATE_IDLE ;
        m_pEvent->Send(0, DMH_EVENT_NO_RESPONSE);
        break;

    case DM_MASTER_STATE_SENDING_SYNC_LOW :  // must have interpacket delay
        m_pUart->GrabBus();
        m_pUart->WriteBaudRate(38400);// restore baud rate
        m_iNetMode = DM_MASTER_STATE_IDLE ;
        m_pEvent->Send(0, DMH_EVENT_SYNC_DONE);

        //Bugzilla 80221 fix
        //Release Cresnet semaphore after sync is definitively complete to prevent transmission overlaps where baud rate gets screwed up
        ReleaseCrestSema();
        break;

    default:
        m_iNetMode = DM_MASTER_STATE_IDLE ;
        m_pEvent->Send(0, DMH_EVENT_FATAL_ERROR_TIMER);
        ReleaseCrestSema() ;
        break;

    }
}
int overruns;
void CDMCresnetUartHost::DmHostUartISR(void)
{
    int intState = m_pUart->CheckRxInterrupt();

    unsigned char ch ;

    // short-circuit when disabled
    if (!m_bEnabled)
        return;

    if (!intState)
        {
            overruns++;

            // Clear overrun condition
            m_pUart->ReadRxBuffer();
            m_pUart->ClearOrInt();
#ifdef DEBUG
            DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_CNET, ERR_CNET_UART_OVERFLOW, overruns);
#else
            DmSystemError(DM_ERROR_LEVEL_WARNING, DM_ERROR_SUBSYS_CNET, ERR_GENERAL_UART_OVERFLOW, overruns);
#endif
        }
    // if it is a receiver interrupt (may be overrun)
    while (intState)
    {
        // short-circuit when disabled
        if (!m_bEnabled)
            return;

        ch = m_pUart->ReadRxBuffer();
        if ( m_iPollState != POLL_STAT_TEST )   //Normal mode
        {
            switch ( m_iNetMode ) {   // check what we are waiting for
//                case DM_MASTER_STATE_PNP_WAIT_FOR_RESPONSE :
                case DM_MASTER_STATE_WAIT_FOR_RESPONSE :
                    m_iNetMode = DM_MASTER_STATE_RECIEVING_LEN ;
                    m_pTimer->ResetTimer() ;
                    m_pPacket.data[3+0] = ch;
                    m_iRxCount = 1 ;        // bump up receive count
                    m_iRxExpected = - 1 ;
                    break;
                case DM_MASTER_STATE_RECIEVING_LEN :
                    m_pPacket.data[3+1] = ch;
                    if ( ch ) {         // non-zero return
                        m_iNetMode = DM_MASTER_STATE_RECIEVING_DATA ;
                        m_iRxExpected = (int)ch + 2 ;
                        m_iRxCount = 2 ;        // bump up receive count
                    }
                    else {              // empty token received
                        m_iNetMode = DM_MASTER_STATE_INTERPACKET_DELAY_AFTER_EMPTY_TOKEN ;
                        m_pUart->GrabBus() ;
                        m_pTimer->SetTimer(THREE_CRESNET_CHAR_TIME);
                        m_iRxCount = 0 ;
                        m_iRxExpected = 0 ;
                        // update DM Link LED timer to inidicate link is alive
                        ManageDMNET_LinkLed();

                    }
                    break;
                case DM_MASTER_STATE_RECIEVING_DATA :
                    m_pPacket.data[3+m_iRxCount++] = ch;
                    if ( m_iRxCount == m_iRxExpected ) {  // if complete, wait for token
                        m_pPacket.data[1] = m_iRxExpected + 1;
                        if (m_pPacket.encapsulated.seg_id != BROADCAST)
                          m_pPacket.encapsulated.seg_id = DM_NET_ID;
                        m_iNetMode = DM_MASTER_STATE_WAIT_FOR_TOKEN_ID ;
                    }
                    break;

                 case DM_MASTER_STATE_WAIT_FOR_TOKEN_ID :
                    m_iNetMode = DM_MASTER_STATE_WAIT_FOR_TOKEN_LEN ;
                    break;

                 case DM_MASTER_STATE_WAIT_FOR_TOKEN_LEN :
                    m_iNetMode = DM_MASTER_STATE_INTERPACKET_DELAY_AFTER_NONEMPTY_TOKEN ;
                    m_iResponded++;
                    m_pUart->GrabBus() ;
                    m_pTimer->SetTimer(ONE_CRESNET_CHAR_TIME);
                    break;

                 default:
                     m_iNetMode = DM_MASTER_STATE_IDLE ;
                     m_pEvent->Send(0, DMH_EVENT_FATAL_ERROR_UART);
                     ReleaseCrestSema() ;
                    break;
            }
        } // if receive error
        // check if another character waiting
        intState = m_pUart->CheckRxInterrupt();
    }
}

void CDMCresnetUartHost::StopPolling(void)
{
    //Bugzilla fix: only set the flag if we need to stop polling
    if(m_iPollState != POLL_STAT_TEST)
    {
       m_PendingStopPoll = true;
    }
    m_iPollState = POLL_STAT_TEST;
    
}

void CDMCresnetUartHost::StartPolling(void)
{
    m_iPollState = POLL_STAT_POLL;
}

void CDMCresnetUartHost::DmHostReleaseNextPacket(void)
{

        if (report_Data_cnt) {
            report_Data_cnt--;
        }
}

CREST_ALL_PACKETS * CDMCresnetUartHost::DmHostGetNextPacket(void)
{
    CREST_ALL_PACKETS *pPtr = 0 ;

    // 2. send data packet if any, add encapsulation if needed
    if (report_Data_cnt)
    {
        // if its a broadcast packet
        if (m_pPacket.data[MASTER_RX_ENCAP_OFFSET] == BROADCAST)
            // don't encapsulate
            pPtr = (CREST_ALL_PACKETS *)&m_pPacket.data[MASTER_RX_ENCAP_OFFSET];
        else // leave encapsulation header
            pPtr = (CREST_ALL_PACKETS *)&m_pPacket.data[0] ;
        }

    return pPtr;
}

void MasterTimer_Isr(UINT8 Timer)
{
    for ( UINT8 i = 0; i < NUM_OF(g_pDmMaster); i++ )
    {
        if ( g_pDmMaster[i] && g_pDmMaster[i]->m_iTimer == Timer )
        {
            g_pDmMaster[i]->DmHostTimerISR();
            return;
        }
    }

    //If made it this far then the timer failed
    DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_GENERAL, ERR_GENERAL_DM_MASTER_INVALID_TIMER_INDEX, Timer);
}

void MasterUart_Isr(UINT8 Uart)
{
    for ( UINT8 i = 0; i < NUM_OF(g_pDmMaster); i++ )
    {
        if ( g_pDmMaster[i] && g_pDmMaster[i]->m_iUart == Uart )
        {
            g_pDmMaster[i]->DmHostUartISR();
            return;
        }
    }

    //If made it this far then the UART failed
    DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_GENERAL, ERR_GENERAL_DM_MASTER_INVALID_UART_INDEX, Uart);
}
void DmHostTaskEntry(UINT32 ptr)
{
    CDMCresnetUartHost *pDmHost = (CDMCresnetUartHost *)(ptr) ;
    pDmHost->DmHostTask();
}

/*
    initalize all hardware and structures for cresnet master port
*/
CDMCresnetUartHost::CDMCresnetUartHost(void):
    m_vSpecialDevHandler(NULL),
    m_iPollState(POLL_STAT_IDLE),
    m_iUart(0),
    m_iTimer(0),
    m_iDmNet(0),
    m_iCurrent(0),
    m_iBusCtrlPin(0),
    m_iDebug(0),
    m_pTxQueue(NULL),
    m_iNetMode(DM_MASTER_STATE_IDLE),
    m_iRxCount(0),
    m_iRxExpected(0),
    m_iTxCount(0),
    m_iTxCurrent(0),
    m_iResponded(0),
    no_response_cnt(0),
    report_Data_cnt(0),
    m_bOnline(0),
    m_ilockId(0),
    m_iTaskId(0),
    m_pTimer(NULL),
    m_pUart(NULL),
    m_pEvent(NULL),
    m_pPollEvent(NULL),
    m_bEnabled(TRUE),
    m_PendingStopPoll(false)
{
    OsCreateLock((UINT32 *)&m_ilockId); // create lock for output buffer

    //Assume no encapsulation checksum helper
    m_pEncapChecksumHelper = 0;
    memset(extra, 0, sizeof(extra));
}

CDMCresnetUartHost::CDMCresnetUartHost(UINT8 DmNetSelect, UINT8 UartSelect, UINT8 TimerSelect, DmEncapChecksumHelper* pHelper /*= 0*/,CDMNetMapper *tmpDMNetMapper /*= pDMNetMapper*/):
    m_vSpecialDevHandler(NULL),
    m_iPollState(POLL_STAT_IDLE),
    m_iUart(0),
    m_iTimer(0),
    m_iDmNet(0),
    m_iCurrent(0),
    m_iBusCtrlPin(0),
    m_iDebug(0),
    m_bEnabled(TRUE),
    m_pTxQueue(NULL),
    m_iNetMode(DM_MASTER_STATE_IDLE),
    m_iRxCount(0),
    m_iRxExpected(0),
    m_iTxCount(0),
    m_iTxCurrent(0),
    m_iResponded(0),
    no_response_cnt(0),
    report_Data_cnt(0),
    m_bOnline(0),
    m_ilockId(0),
    m_iTaskId(0),
    m_pTimer(NULL),
    m_pUart(NULL),
    m_pEvent(NULL),
    m_pPollEvent(NULL),
    m_pDMNetMapper(tmpDMNetMapper),
    m_PendingStopPoll(false)
{
    OsCreateLock((UINT32 *)&m_ilockId); // create lock for output buffer

    //Save the encapsulation checksum helper
    m_pEncapChecksumHelper = pHelper;

    if(!m_pDMNetMapper)
    {
        DmSystemError(DM_ERROR_LEVEL_FATAL,DM_ERROR_SUBSYS_CNET,ERR_CNET_DMNET_MAPPER_UNINITIALIZED, 1);
    }
    memset(extra, 0, sizeof(extra));

    SetUartAndTimer(DmNetSelect, UartSelect, TimerSelect);
}

void CDMCresnetUartHost::SetUartAndTimer(UINT8 DmNetSelect, UINT8 UartSelect, UINT8 TimerSelect)
{
    m_iPollState = POLL_STAT_IDLE ;
    m_iTaskId = 0;
    m_iDebug = 0;
    report_Data_cnt = 0;
    m_vSpecialDevHandler= NULL;
    m_pTxQueue = NULL;

#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT
    m_iDmNet = DmNetSelect;
#else
	if (DmNetSelect == 1) {
        m_iDmNet = 1;
    }
    else {
        m_iDmNet = 2;
    }
#endif
    
    m_iUart = UartSelect;
        AssignUartISR(m_iUart, MasterUart_Isr) ;
    m_iTimer = TimerSelect ;
        AssignTimerISR(m_iTimer, MasterTimer_Isr) ;
    m_iNetMode = DM_MASTER_STATE_IDLE ;
}

void CDMCresnetUartHost::Init(void)
{
    no_response_cnt = REPORT_IF_MISSING_OR_IF_PRESENT ;

    m_pPacket.data[0] = CNET_ID_CTRLSYS;
    m_pPacket.data[1] = 0;
    m_pPacket.data[2] = CRESNET_ENCAPSULATED;

    char taskName[5];

    sprintf(taskName, "Dm%d",m_iDmNet);

    m_iTaskId = OsCreateNamedPriorityTask(
                      DmHostTaskEntry, // entry point
                      0,                         // period
                      0, // CRESNET_MASTER_RX_QUEUE_ITEMS,
                      (UINT32 *)this ,
                      &extra[0],
                      (signed char const *)taskName,
                      CRESNET_HOST_TASK_PRIORITY, // priority
                      0   // stack size
                      ) ;

}

void CDMCresnetUartHost::SetDefaults(void)
{
}

void CDMCresnetUartHost::ResetState(void)
{
}

INT8 CDMCresnetUartHost::SendPacket(CREST_ALL_PACKETS *packet)
{
    // short-circuit if not selected
    if (!m_bEnabled)
        return NU_SUCCESS;

    if (ObtainCrestSema(HOST_SEMAPHORE_TOUT) != NU_SUCCESS)
    {
       #ifdef USE_DM_NET_DEBUG
        if (m_iDebug ) // DMH_DEB_ERRORS_BIT
        {
           DmConsolePrintf("DMh%d TOUT3\r", m_iDmNet);
       }
       #endif
        return NU_UNAVAILABLE ;
    }

    if (IsDmFieldDebugIndexActive(DM_FIELD_CRESNET_TX_IDX))
    {    
        char txName[5];
        sprintf(txName, "TX%d",m_iDmNet);
        
        print_packet((char *)&packet->data[0], txName);  
        
    }


    switch (packet->generic.cmd)
    {
    case TJI_CNTRL_REQ:
    case RCON_CMD: // add it to rcon manager poll list
//        m_pHostTask->m_pRconHandler->AddToPoll((int)packet->rcon.net_id);
        break;

    case INTERNAL: // check for internal packets, they should not go to the wire
//        ProcessControlPacket(packet) ;
        break;
    }

    m_pTimer->ResetTimer();

    m_iNetMode = DM_MASTER_STATE_BEFORE_SENDING_DATA ;
    m_iTxCount =  (UINT16)(packet->generic.len) + 2 ;
    m_iTxCurrent = 0;
    memcpy( (void *)&m_pTxPacket.data[0],
            (void *)packet,
            (size_t)m_iTxCount );

    //Check for the encapsulation helper class
    if ( m_pEncapChecksumHelper )
    {
        UINT8 bNeedChecksum;
        UINT8 bUpdateTxSequence;
        CREST_ALL_PACKETS* pRawPacket = &m_pTxPacket;

        //Check if we need to look for a name request
        if ( m_pEncapChecksumHelper->IsNameRequestTimerNeeded() && m_pDMNetMapper->IsStandardFlow() )
        {
            //Check if this is a name request packet
            if ( (packet->generic.len == 0x02) &&
                 (packet->generic.cmd == CNET_COMMAND) &&
                 (packet->generic.data[0] == CNET_CMD_NAME_REQ) )
            {
                m_pEncapChecksumHelper->StartNameRequestTimer();
            }
        }

        //Check if we need checksum encapsulation
        bNeedChecksum = m_pEncapChecksumHelper->IsEncapsulatedChecksumEnabled();

        //Check for a sequence request
        if ( pRawPacket->generic.len == (MAX_DM_SEQUENCE_REQUEST_PACKET_BYTES - 2) &&
             pRawPacket->generic.cmd == DM_SEQUENCE_REQUEST_PACKET_CMD )
        {
            //Calculate the UpdateTxSequence
            if ( bNeedChecksum )
            {
                bUpdateTxSequence = m_pEncapChecksumHelper->GetNextSequence(m_pEncapChecksumHelper->m_bLocalTxSequence);
            }
            else
            {
                bUpdateTxSequence = m_pEncapChecksumHelper->m_bLocalTxSequence;
            }

            //Bugzilla 44193 fix
            //Update the sequence numbers at the latest possible time
            pRawPacket->DmSequenceRequestPacket.TxSequence = bUpdateTxSequence;
            pRawPacket->DmSequenceRequestPacket.RxSequence = m_pEncapChecksumHelper->m_bLocalRxSequence;
        }

        //Check if encapsulation is needed
        if ( bNeedChecksum )
        {
            if ( m_pEncapChecksumHelper->CreateEncapsulatedChecksumPacket((UINT8*)&m_pTxPacket.data[0], CNET_MAX_MSG_SIZE) )
                m_iTxCount += DM_ENCAPSULATED_CHECKSUM_PACKET_ADDITIONAL_BYTES;
        }
    }

    m_pUart->GrabBus();
    m_pTimer->SetTimer(ONE_CRESNET_CHAR_TIME);

    return NU_SUCCESS;
}

INT8 CDMCresnetUartHost::SendToken(UINT8 cnetId)
{
    m_iCurrent = cnetId ;

    m_pUart->GrabBus() ;

    m_iNetMode = DM_MASTER_STATE_DELAY_BEFORE_TOKEN_ID ;
    m_iRxCount = 0 ;
    m_iRxExpected = -1 ;
    m_iResponded = 0;  // clear RESPONDED flag before sending token

    m_pTimer->SetTimer(ONE_CRESNET_CHAR_TIME);

    return NU_SUCCESS;
}

INT8 CDMCresnetUartHost::SendSync(void)
{
    if (ObtainCrestSema(HOST_SEMAPHORE_TOUT) != NU_SUCCESS) {
       #ifdef USE_DM_NET_DEBUG
       if (m_iDebug ) { // DMH_DEB_ERRORS_BIT
           DmConsolePrintf("DMh%d TOUT4\r", m_iDmNet);
       }
       #endif
        return NU_UNAVAILABLE ;
    }

    m_iNetMode = DM_MASTER_STATE_SENDING_SYNC_LOW ;
    m_pUart->GrabBus() ;
    m_pUart->WriteBaudRate(19200);// set baud to 19200
    m_pUart->WriteTxBuffer(0x80);
    m_pTimer->SetTimer(TWO_AND_HALF_CHAR_TIME*3);

    m_pEvent->Receive(DM_WAIT_FOR_SYNC_END_MS); // consume event

    //Bugzilla 80221 fix
    //Allow the timer ISR to release the semaphore in case the event takes longer than expected or multiple events are received

    return NU_SUCCESS;
}

/**
 * \author        Gennady
 * \brief         Locks the critical region;  don't call from isr since it waits forever till region
 *                is available
 * \date          7/28/08
 * \param         pHandle - pointer to Semaphore's handle
 * \return        INT32
 * \retval        0 if success; -1 if in ISR or semaphore not found;
**/
INT8 CDMCresnetUartHost::ObtainCrestSema(UINT32 wait)
{
    return (INT8)OsLockWait(m_ilockId, wait);
}

/**
 * \author        Gennady
 * \brief         unlocks the critical region;
 * \date          7/28/08
 * \param         pHandle - pointer to Semaphore's handle
 * \return        INT32
 * \retval        returns 1 if some task woke up
**/
INT8 CDMCresnetUartHost::ReleaseCrestSema(void)
{
    return (INT8)OsUnlock(m_ilockId);
}

/**
 * \author      Larry Salant
 * \date        1/22/2009
 * \brief       called when the Control flow is reversed: the master is getting
 *              packets from the switch.  Send the packets to the Cresnet parser
 *              for processing
 * \param       void
 * \return      UINT32
 * \retval      flag if immediate response to a packet is required
 * \note        This routine is called from the main loop of the cresnet host task (not the
 *              isr) if a packet was received by the ISR.  It is called from within the
 *              Host's critical region.
 */
UINT32 CDMCresnetUartHost::MasterMidpointRxIsr(void)
{
  CREST_ALL_PACKETS * packetPtr;
  UINT32 command = 0;
  UINT16 byteCnt;
  UINT8 *pBuffer;

  // the packet was offset into the buffer when it was received
  // (to be able to encapsulate when in normal flow)
  packetPtr = (CREST_ALL_PACKETS *)&m_pPacket.data[MASTER_RX_ENCAP_OFFSET] ;
  
  // if our id or a broadcast packet, perform any immediate processing required
  if ( (packetPtr->generic.net_id == BROADCAST) || packetPtr->generic.net_id == DM_NET_ID)
  {
    // Check to see if it is an device ID request
    if ((packetPtr->generic.len == 0x02) && (packetPtr->generic.cmd == CNET_COMMAND)
      && (packetPtr->generic.data[0] == CNET_CMD_NAME_REQ) )   // device ID request
    {
      command = CNET_CMD_NAME_REQ;
    }

    // Send packet to Cresnet parser for other processing
    // get a temporary buffer big enough for the message (+2 is for id,len bytes)
    byteCnt = packetPtr->generic.len + 2;
    // add one more character incase someone adds a null terminator.
    if (MemMgr && (pBuffer = MemMgr->GetBlock(byteCnt+1)) != 0)
    {
      // copy message to buffer;  This buffer will be released after the command is
      // processed by the cresnet task.
      memcpy((void*)&pBuffer[CRESNET_BUF_ID_OFFSET], packetPtr, byteCnt);

      // indicate the source for the parser
      if(packetPtr->generic.net_id == BROADCAST)
        pBuffer[CRESNET_BUF_SRC_OFFSET] = NET_SOURCE_CRESNET_BROADCAST;
      else
        pBuffer[CRESNET_BUF_SRC_OFFSET] = NET_SOURCE_CRESNET;

      // funnel to common processing
      CresnetSlaveAcceptIncomingPacket(pBuffer);

      // Since we are in reverse mode, and the controller sent us something, if this was
      // not a command which we need to respond to immediately, do another poll immediately
      // incase there is more data coming (e.g. upload)
      if (command == 0)
         command = POLL_AGAIN_IMMEDIATE;
    }
    // error buffer not available ...
    // for now just reset buffer and throw out message
  }
  return command;
}

INT8 CDMCresnetUartHost::DmProccessPacketRx(void)
{
    #ifdef USE_DM_NET_DEBUG
    if (m_iDebug & DMH_DEB_PROCES_PACKET_RX_BIT) {
        DmConsolePrintf("DMh%d: Rcv=%d, Exp=%d\r", m_iDmNet, m_iRxCount, m_iRxExpected);
    }
    #endif

    // add one encapsulation level and send it to corresponding slave port

    report_Data_cnt++;
    if(m_vSpecialDevHandler)
        m_vSpecialDevHandler((void *)NULL, 0);

    return 0;
}

/**
 * \author      Larry Salant
 * \date        1/22/2009
 * \brief       called when the Control flow is reversed: the master is getting
 *              packets from the switch.  Send the packets to the Cresnet parser
 *              for processing
 * \param       immediate commmand to send (e.g. device id); 0 if get from queue, if any
 * \return      void
 * \retval      none
 * \note        This routine is called from the main loop of the cresnet host task so it's not in
 *              the host's critical region
 */
void CDMCresnetUartHost::DmProccessPacketTx(UINT8 Command)
{
    // if device id is requested,
    if (Command == CNET_CMD_NAME_REQ)
    {
      UINT8 length;
      UINT8 *idmsg;

      // get temporary buffer to build the message
      if (MemMgr && (idmsg = MemMgr->GetBlock(CNET_MAX_MSG_SIZE)) != 0)
      {
        /* The 5th character is the start of the actual string */
        FirmwareGetName((char *)&idmsg[5], CNET_MAX_MSG_SIZE-6, STREAM_1);

        length = strlen((char *)&idmsg[5]) +  3;

        /* now fill in the beginning of the packet */
        idmsg[0]= CNET_ID_CTRLSYS;
        idmsg[1]= length;
        idmsg[2]= NAME_REQ;
        idmsg[3]= 0;
        idmsg[4]= 0;
        SendPacket((CREST_ALL_PACKETS *)idmsg);
        MemMgr->FreeBlock(idmsg);
      }
    }
    // if no immediate command, see if there's a packet queue'd
    else if (m_pTxQueue != NULL)
    {
      // send the packet
      SendPacket(m_pTxQueue);
      // empty the queue (it's copied into the tx buffer)
      MemMgr->FreeBlock((UINT8*)m_pTxQueue);
      m_pTxQueue = NULL;
    }

}
/**
 * \author      Larry Salant
 * \date        1/31/2009
 * \brief       if the Queue for the master port (in reverse control
 *              flow) is empty, queue message  (called by Join State to queue packets)
 * \param       pointer to packet to queue
 * \param       length of packet
 * \return      bool
 * \retval      true if packet sent
 */
bool CDMCresnetUartHost::HostEnqueueOutgoingPacket(CREST_ALL_PACKETS *pPacket, UINT16 lLength)
{
  bool sent = false;

  // if queue is empty
  if (!m_pTxQueue)
  {
    // get temporary buffer
    if (MemMgr && (m_pTxQueue = (CREST_ALL_PACKETS *)MemMgr->GetBlock(lLength+1)) != 0)
    {
      // copy message to buffer;  This buffer will be released after the command is sent
      memcpy(m_pTxQueue, pPacket, lLength);
      sent = true;
    }
    else
        DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_CNET, ERR_CLIBRARY_NO_MEMORY, 0);
  }
  return sent;
}
/**
* \author      Larry Salant
* \date        3/5/2009
* \brief       Indicates if there is an output message pending
* \param       void
* \return      bool
* \retval      true if queue is empty
*/
bool CDMCresnetUartHost::OutputQueueAvailable(void)
{
    return !m_pTxQueue;
}

/*
Dm net is point-to-point, cnet ID is fixed to DM_NET_ID
*/
INT8 CDMCresnetUartHost::PollDevice(void)
{
    INT8 result = 0;
    bool bFlowChange;
    UINT8 bPacketIntercepted = false;
    UINT8* pBuffer;
    UINT16 sByteCount = 0;

    // if in standard control flow, check if slave side removed peviously accumulated packets
    if (m_pDMNetMapper->IsStandardFlow() && report_Data_cnt)
    {
        return NU_UNAVAILABLE ;
    }

    if ( ObtainCrestSema(HOST_SEMAPHORE_TOUT) != NU_SUCCESS )
    {
#ifdef USE_DM_NET_DEBUG
        if (m_iDebug ) // DMH_DEB_ERRORS_BIT
        {
            DmConsolePrintf("DMh%d TOUT1\r", m_iDmNet);
        }
#endif
        return NU_UNAVAILABLE ;
    }

    m_pEvent->Receive(0); // consume event

    SendToken(DM_NET_ID) ;

    if (m_pEvent->Receive(DM_WAIT_FOR_RESPONSE_MS) == 0)
    { // consume event
#ifdef USE_DM_NET_DEBUG
        if (m_iDebug & DMH_DEB_EVENTS_BIT)
        {
            DmConsolePrintf("Dmh%d Evt:%d\r", m_iDmNet, m_pEvent->m_iReceved);
        }
#endif

        switch (m_pEvent->m_iReceved)
        {
            case DMH_EVENT_DATA_RECEIVED:
                switch ( no_response_cnt )
                {

                    case REPORT_IF_MISSING_OR_IF_PRESENT:
                        DmSetOnLineStatus(DIGI_OFF);  // mark it as ONLINe
                        break;

                    case MISSING_DEVICE_POLLS:  // was missing before, do nothing
                        DmSetOnLineStatus(DIGI_OFF);  // mark it as ONLINe
                        break;

                }
                no_response_cnt = 0;

#ifdef USE_DM_NET_DEBUG
                if (m_iDebug & DMH_DEB_RX_BIT)
                    print_packet((char *)&m_pPacket.data[0], (m_iDmNet==1) ? "Rx1" : "Rx2");
#endif

                //Check if we need to externally process the packet
                if ( pfMasterProcessPacket )
                {
                    pfMasterProcessPacket(&m_pPacket.data[MASTER_RX_ENCAP_OFFSET]);
                }

                //Assume packet not intercepted
                bPacketIntercepted = false;

                //Check for the encapsulation checksum helper class
                if ( m_pEncapChecksumHelper )
                {
                    //Check for an encapsulated checksum packet
                    if ( m_pEncapChecksumHelper->IsEncapsulatedChecksumPacket( (UINT8*)&m_pPacket.data[MASTER_RX_ENCAP_OFFSET], sizeof(CREST_ALL_PACKETS) ) )
                    {
                        UINT8 bStatus;
                        bStatus = m_pEncapChecksumHelper->VerifyPacket( (UINT8*)&m_pPacket.data[MASTER_RX_ENCAP_OFFSET], sizeof(CREST_ALL_PACKETS) );
                        switch ( bStatus )
                        {
                            case DM_ENCAPSULATED_PACKET_STATUS_NO_ERRORS:
                            case DM_ENCAPSULATED_PACKET_STATUS_ERROR_MISMATCH_SEQUENCE:
                                //Strip the packet
                                m_pEncapChecksumHelper->RemoveEncapsulatedChecksumPacket( (UINT8*)&m_pPacket.data[MASTER_RX_ENCAP_OFFSET], sizeof(CREST_ALL_PACKETS) );
                                //adjust the encapsulated packet length accordingly
                                m_pPacket.generic.len -= DM_ENCAPSULATED_CHECKSUM_PACKET_ADDITIONAL_BYTES;
                                break;

                            case DM_ENCAPSULATED_PACKET_STATUS_ERROR_WRONG_TYPE:
                            case DM_ENCAPSULATED_PACKET_STATUS_ERROR_INVALID_PARAMETERS:
                            case DM_ENCAPSULATED_PACKET_STATUS_ERROR_INVALID_COUNT:
                            case DM_ENCAPSULATED_PACKET_STATUS_ERROR_BAD_CHECKSUM:
                                //Log an error b/c we will throw it away
                                DmSystemError( DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_CNET, ERR_CNET_ENCAP_RX_DROP_PKT, bStatus );

                                //Set the packet intercepted flag and do nothing with the packet
                                bPacketIntercepted = true;
                                break;
                        }
                    }

                    //Bugzilla 52637 fix
                    //Name request timer should check after encapsulation is removed
                    //Check if we need to look for a name request
                    if ( m_pEncapChecksumHelper->IsNameRequestTimerNeeded() && m_pDMNetMapper->IsStandardFlow() )
                    {
                        if ( (m_pPacket.data[MASTER_RX_ENCAP_OFFSET+1] > 3) &&          //LEN
                             (m_pPacket.data[MASTER_RX_ENCAP_OFFSET+2] == NAME_REQ) )   //CMD
                        {
                            //Stop the name request timer
                            m_pEncapChecksumHelper->StopNameRequestTimer();
                        }
                    }

                    //Look for a sequence request or sequence response packet
                    if ( !bPacketIntercepted &&
                         ( (m_pPacket.encapsulated.seg_len == (MAX_DM_SEQUENCE_REQUEST_PACKET_BYTES - 2)) ||
                           (m_pPacket.encapsulated.seg_len == (MAX_DM_SEQUENCE_RESPONSE_PACKET_BYTES - 2)) ||
                           (m_pPacket.encapsulated.seg_len == (MAX_DM_LONG_REACH_PACKET_BYTES - 2)) ) )
                    {
                        switch ( m_pPacket.encapsulated.seg_cmd )
                        {
                            case DM_SEQUENCE_REQUEST_PACKET_CMD:
                            case DM_SEQUENCE_RESPONSE_PACKET_CMD:
                            case DM_LONG_REACH_PACKET_CMD:
                                //Force the ID so we process it correctly
                                m_pPacket.encapsulated.seg_id = m_pEncapChecksumHelper->GetPacketSlaveId();

                                //Set the packet intercepted flag
                                bPacketIntercepted = true;

                                //Send to the local slave parser
                                sByteCount = m_pPacket.encapsulated.seg_len + 2;
                                if (MemMgr && (pBuffer = MemMgr->GetBlock(sByteCount+2)) != 0)
                                {
                                    // copy message to buffer;  This buffer will be released after the command is
                                    // processed by the cresnet task.
                                    memcpy((void*)&pBuffer[CRESNET_BUF_ID_OFFSET], (UINT8*)&m_pPacket.data[MASTER_RX_ENCAP_OFFSET], sByteCount);

                                    // indicate the source for the parser
                                    pBuffer[CRESNET_BUF_SRC_OFFSET] = NET_SOURCE_CRESNET;

                                    // funnel to common processing
                                    CresnetSlaveAcceptIncomingPacket(pBuffer);
                                }
                                break;
                        }
                    }
                }

                // check if it's a broadcast command for Digital Media Control Flow Direction Message
                // <id> <cnt> <32> <Flags>
                if ((m_pPacket.encapsulated.seg_id == BROADCAST) &&
                    (m_pPacket.encapsulated.seg_cmd == DM_DEV_CONTROL_FLOW_DIR_CMD))
                {
                    // This packet is sent by the switch firmware to all the DM Links under the DM Input Card requesting to
                    // reverse/normalize the control flow direction.  All Repeaters / DGE / WP / DM Output Card will accept this message,
                    // act on it and pass it on. No DM devices will respond to this message. If a DM Output Card receives this message,
                    // it should just pass this message onto the DM Switch firmware.
                    // The <id> will always be a BroadCast ID (0xFF).
                    // <Flags> are defined as: 01: Reverse Control Flow; 02: Normal Control Flow
                    // clear any pending messages

                    //Assume no flow change
                    bFlowChange = false;

                    if (m_pPacket.encapsulated.data[0] == DM_DEV_CONTROL_FLOW_REVERSE)
                    {
                        bFlowChange = m_pDMNetMapper->SetControlFlowReverse();

                        //Make sure the name request timer is disabled on reverse mode
                        if ( m_pEncapChecksumHelper && m_pEncapChecksumHelper->IsNameRequestTimerNeeded() )
                        {
                            m_pEncapChecksumHelper->StopNameRequestTimer();
                        }
                    }
                    else // normal flow
                    {
                        bFlowChange = m_pDMNetMapper->SetControlFlowNormal();
                    }

                    //Bugzilla fix
                    //This fixes DM6X1 to output card race condition where broadcast packets would get sent as encapsulated
                    //Check for a flow change
                    if ( bFlowChange )
                        report_Data_cnt = 0;
                }

                if ( !bPacketIntercepted )
                {
                    result = m_pDMNetMapper->ProcessRxPacket(DMNET_MASTER, m_iDmNet-1);
                }
                else
                {
                    result = 0;
                }
                break;

            case DMH_EVENT_EMPTY_TOKEN:
                switch ( no_response_cnt )
                {

                    case REPORT_IF_MISSING_OR_IF_PRESENT:
                        DmSetOnLineStatus(DIGI_OFF);  // mark it as ONLINe
                        break;

                    case MISSING_DEVICE_POLLS:  // was missing before, do nothing
                        DmSetOnLineStatus(DIGI_OFF);  // mark it as ONLINe
                        break;

                }
                no_response_cnt = 0;
                break;

            case DMH_EVENT_NO_RESPONSE:
                switch ( no_response_cnt )
                {

                    case REPORT_IF_MISSING_SET_MASK_BIT_IF_PRESENT :
                    case 0 :                // first time no response
                        no_response_cnt = 1 ;
                        //DmConsolePrintf("Dmh%d Missed first Poll \r", m_iDmNet);
                        break;

                    case REPORT_IF_MISSING_OR_IF_PRESENT:
                        no_response_cnt = MISSING_DEVICE_POLLS ;
                        DmSetOnLineStatus(DIGI_ON);  // mark it as OFFLINe
                        break;

                    case MISSING_DEVICE_POLLS:  // was missing before, do nothing
                        break;

                    default: // bump up no_resonse_cnt, report OFFLINE if missed enough
                        no_response_cnt++ ;
                        if ( no_response_cnt >= MISSING_DEVICE_POLLS )
                        { // report OFFLINE
                            no_response_cnt = MISSING_DEVICE_POLLS ;
                            DmSetOnLineStatus(DIGI_ON);  // mark it as OFFLINe
                        }
                        break;

                }
                break;

            case DMH_EVENT_FATAL_ERROR_UART:
                DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_GENERAL, ERR_GENERAL_DM_MASTER_POLL_DEVICE_UART_ERROR, m_iDmNet);
                break;

            default:
                DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_GENERAL, ERR_GENERAL_DM_MASTER_POLL_DEVICE_UNKNOWN_EVENT, m_iDmNet);
                break;
        }
    }
    else
    { // huge problem: error
#ifdef USE_DM_NET_DEBUG
        if (m_iDebug )  // DMH_DEB_ERRORS_BIT
            DmConsolePrintf("DMh%d TOUT2\r", m_iDmNet);
#endif
    }

    ReleaseCrestSema();

    return result;
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       queue task to handle parsing incoming msgs
 * \return      none
 * \retval      void
 * \param       param - pointer to the cresnet packet
 */
void CDMCresnetUartHost::DmHostTask(void)
{
    StartPolling();

    // main polling loop
    UINT8 event = 0, result=0 ;

    m_pPollEvent->Receive(0); // initial event clear
    event = m_pPollEvent->m_iReceved ;

    if (IsDmFieldDebugIndexActive(DM_FIELD_CRESNET_TX_IDX))
        DmConsolePrintf(" DmHostStart(%d): timer %d, uart %d\r",m_iDmNet, m_pTimer->m_iNumber, m_pUart->m_iNumber);

    event = 0;

    while (1)
    {
       // unless an immediate poll is requested, wait before doing the next poll
       if (event != POLL_AGAIN_IMMEDIATE)
         HwDelayMsec(MASTER_POLL_TIME);


       #ifdef USE_DM_NET_DEBUG
       if (m_iDebug & DMH_DEB_EVENTS_BIT) {
           if (result == 0) // event received
               DmConsolePrintf("Dmh%d Evt:%d\r", m_iDmNet, event);
       }
       #endif


        if (m_iPollState == POLL_STAT_POLL)
        {
          // poll downstream device and read back any data from it,  send it
          event = PollDevice();
          // if there is data to send (only happens in reverse control flow)
          m_pDMNetMapper->QueFromMasterToController(this,event);
        }
        else if (m_iPollState == POLL_STAT_TEST)
        {
            event = 0;
            if ( m_bOnline )
            {
                //Initialize offline
                DmSetOnLineStatus(DIGI_ON);
            }

            //Reset all variables
            m_pEvent->Receive(0); // consume event
            no_response_cnt = MISSING_DEVICE_POLLS;
            m_PendingStopPoll = false;
        }
        else
        {
            event = 0;
        }

        //Only send syncs if we are polling
        if (m_iPollState != POLL_STAT_TEST)
        {
            if ( !(result++) )
            {
            	SendSync();
        	}
    	}
	}
}

// parameter :  DIGI_ON if going OFFLINE,  DIGI_OFF if OnlIne
void CDMCresnetUartHost::DmSetOnLineStatus(UINT8 state)
{
    if (state == DIGI_ON)
    {
        m_bOnline = FALSE;
    }
    else
    {
        m_bOnline = TRUE;
    }

    // send the online status command
	 m_pDMNetMapper->MasterStatusChange(m_iDmNet-1,m_bOnline);

#ifdef USE_DM_NET_DEBUG
    if (m_iDebug)
    {
    DmConsolePrintf("Dmh%d line:%X(%d)\r", m_iDmNet, state, m_bOnline);
    }
#endif
    if (m_vSpecialDevHandler)
        m_vSpecialDevHandler((void *)NULL, 1);

    if (pfMasterOnlineStatus)
        pfMasterOnlineStatus(m_bOnline);

    //Bugzilla 44329 fix
    if ( m_pEncapChecksumHelper )
    {
        //Notify the encapsulation checksum helper the UART status
        m_pEncapChecksumHelper->SendUartOnlineStatus(m_bOnline ? true : false);
    }
}
/**
 * \author      Larry Salant
 * \date        1/31/2009
 * \brief       returns whether or not the device connected to master port is online
 * \param       void
 * \return      bool
 * \retval      true if device on line
 */
bool CDMCresnetUartHost::IsDeviceConnected(void)
{
  return m_bOnline;
}

INT32 ReportDMnetCmd(UINT32 argc, char * cmd)
{
    return 0;
}

/*
INT32 SendDMnetPacketCmd(UINT32 argc, char * cmd)
{
    return 0;
}
*/

/**
 * \author      Chris Merck
 * \date        25 June 2013
 * \brief       Reconfigures after UART has been used for another purpose.
 * \param       bEnable: TRUE = restore UART etc to working condition
 *                       FALSE = stop using UART
 * \return      bool
 * \retval      true if device on line
 */
void CDMCresnetUartHost::Reconfigure(BOOL bEnable)
{
    // NOTE: let's see if we can get away without disabling the timer
    UINT16 timeout;

    //Bugzilla fix: only process state changes
    if(bEnable == m_bEnabled)
        return;

    if (bEnable)
    {
    	OsEnterCritical();
    
        // re-enable access to HW, reassign ISR and comspec
        AssignUartISR(m_iUart, MasterUart_Isr) ;
        m_pUart->Init();
        m_pTimer->Init();
        m_bEnabled = TRUE;
        StartPolling();
        if(m_iTaskId)
        {
            OsResumeAdvTask(m_iTaskId);
        }
        else
        {
            Init();
        }

        OsExitCritical();
    }
    else
    {
        //Check if the task has already been created
        if ( m_iTaskId )
        {
            //Tell state machine to stop polling
	        StopPolling();
	
	            //Wait for class to process the stop polling call
	        timeout = WAIT_FOR_STOP_POLL_TIME;
	        while ( m_PendingStopPoll && timeout )
	        {
	            HwDelayMsec(WAIT_FOR_STOP_POLL_RATE);
	            timeout--;
	        }
	        
	            //Log error if we timed out waiting for the stop polling
	        if(m_PendingStopPoll)
	        {
                DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_GENERAL, ERR_GENERAL_DM_MASTER_RECONFIGURE_STOP_POLLING_TIMEOUT, m_iDmNet);
            }
        }

        //Bugzilla 80221 fix
        //Grab the host semaphore to ensure that we can park the state machine properly
        BOOL bSemaphore = FALSE;
        if ( ObtainCrestSema(CRESNET_HOST_RECONFIGURE_SEMAPHORE_TIMEOUT_MS) == NU_SUCCESS )
        {
            bSemaphore = TRUE;
        }
        else
        {
            DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_GENERAL, ERR_GENERAL_DM_MASTER_RECONFIGURE_DISABLE_SEMAPHORE_TIMEOUT, m_iDmNet);
        }

        //Bugzilla 80221 fix
        //Disable the class only when polls have stopped and we secured the semaphore
        m_bEnabled = FALSE;
        if ( bSemaphore )
        {
            ReleaseCrestSema();
        }

        OsEnterCritical();

        if(m_iTaskId)
        {
            OsSuspendAdvTask(m_iTaskId);
        }
        m_pTimer->DeInit();
        m_pUart->DeInit();
    
    	OsExitCritical();
    }
}

#undef _CRESNET_HOST_CPP_


